<?php
// 경로/파일: /var/www/html/elfinder/php/connector.minimal.php
// 설명:
// - /etc/realnas/elfinder.json 에서 roots를 읽는다. (1순위)
// - (폴백) /opt/realcom-nas/config/elfinder.json 도 지원한다. (2순위)
// - roots는 UI(disk.html)와 맞게 {alias, path, url} 를 기본 지원한다.
// - 추가 호환: {web_path} 지원
// - url이 없으면 "현재 접속 도메인 + basePath + /web_path" 로 자동 생성 가능(옵션).
// - 숨김파일(.) + lost+found 접근 차단 (UI 전용)
// - X-Forwarded-Proto / X-Forwarded-Host 지원 (nginx reverse proxy 환경)
// - roots가 0개여도 무조건 fallback 1개 만들어서 errNoVolumes/500 방지

error_reporting(0);

is_readable(__DIR__ . '/vendor/autoload.php') && require __DIR__ . '/vendor/autoload.php';
require __DIR__ . '/autoload.php';

elFinder::$netDrivers['ftp'] = 'FTP';

// -------------------------------------------------
// 접근 제어 (중요)
// -------------------------------------------------
function access($attr, $path, $data, $volume, $isDir, $relpath) {
    $basename = basename($path);

    // 1) 숨김파일/폴더 (.xxx) 차단
    if (strlen($basename) > 0 && $basename[0] === '.' && strlen($relpath) !== 1) {
        return !($attr === 'read' || $attr === 'write');
    }

    // 2) lost+found 차단 (리눅스 시스템 폴더, UI에서만 숨김)
    if ($basename === 'lost+found') {
        return !($attr === 'read' || $attr === 'write');
    }

    return null;
}

// -------------------------------------------------
// util
// -------------------------------------------------
function norm_path($p) {
    $p = rtrim((string)$p, "/");
    return $p === '' ? '/' : $p;
}

function detect_scheme() {
    if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO'])) {
        $p = explode(',', $_SERVER['HTTP_X_FORWARDED_PROTO'])[0];
        $p = trim($p);
        if ($p === 'http' || $p === 'https') return $p;
    }
    if (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') return 'https';
    return 'http';
}

function detect_host() {
    if (!empty($_SERVER['HTTP_X_FORWARDED_HOST'])) {
        $h = explode(',', $_SERVER['HTTP_X_FORWARDED_HOST'])[0];
        return trim($h);
    }
    if (!empty($_SERVER['HTTP_HOST'])) return $_SERVER['HTTP_HOST'];
    if (!empty($_SERVER['SERVER_NAME'])) return $_SERVER['SERVER_NAME'];
    return 'localhost';
}

function script_base_url() {
    $scheme = detect_scheme();
    $host   = detect_host();
    $script = isset($_SERVER['SCRIPT_NAME']) ? $_SERVER['SCRIPT_NAME'] : '';
    $basePath = preg_replace('#/php/[^/]+\.php$#', '', $script);
    $basePath = rtrim($basePath, '/');
    return $scheme . '://' . $host . $basePath;
}

function load_config($path) {
    if (!is_readable($path)) return null;
    $raw = @file_get_contents($path);
    if ($raw === false) return null;
    $json = @json_decode($raw, true);
    if (!is_array($json)) return null;
    return $json;
}

function safe_web_path($s) {
    $s = trim((string)$s);
    $s = trim($s, "/");
    $s = str_replace(array('..', '\\', '/'), '', $s);
    $s = preg_replace('/\s+/', '-', $s);
    $s = preg_replace('/[^a-zA-Z0-9._-]/', '', $s);
    return $s;
}

function build_local_root($alias, $path, $url) {
    $path  = norm_path($path);
    $alias = trim((string)$alias);

    $root = array(
        'driver'        => 'LocalFileSystem',
        'path'          => $path . '/',
        'alias'         => $alias !== '' ? $alias : basename($path),
        'uploadDeny'    => array(),
        'uploadAllow'   => array('all'),
        'uploadOrder'   => array('deny', 'allow'),
        'accessControl' => 'access',
        'archive'       => array(
            'create'  => array('application/zip'),
            'extract' => array('application/zip'),
        ),
    );

    $url = trim((string)$url);
    if ($url !== '') {
        $root['URL'] = rtrim($url, '/') . '/';
    }

    return $root;
}

function pick_conf_path() {
    // 1) 정책상 1순위: /etc/realnas/elfinder.json
    $p1 = '/etc/realnas/elfinder.json';
    if (is_readable($p1)) return $p1;

    // 2) 폴백: /opt/realcom-nas/config/elfinder.json (yml/app 정책과 동일)
    $p2 = '/opt/realcom-nas/config/elfinder.json';
    if (is_readable($p2)) return $p2;

    // 3) 둘 다 없으면 1순위 경로를 기본으로 리턴(로그/운영 상 위치 고정)
    return $p1;
}

// -------------------------------------------------
// 설정 정책
// -------------------------------------------------
define('AUTO_URL', true);

// -------------------------------------------------
// roots 구성
// -------------------------------------------------
$CONF_PATH = pick_conf_path();
$conf = load_config($CONF_PATH);

$roots   = array();
$baseUrl = script_base_url();

if (is_array($conf) && isset($conf['roots']) && is_array($conf['roots']) && count($conf['roots']) > 0) {
    foreach ($conf['roots'] as $r) {
        if (!is_array($r)) continue;

        $alias = isset($r['alias']) ? trim((string)$r['alias']) : '';
        $path  = isset($r['path'])  ? trim((string)$r['path'])  : '';
        $url   = isset($r['url'])   ? trim((string)$r['url'])   : '';
        $webp  = isset($r['web_path']) ? trim((string)$r['web_path']) : '';

        if ($path === '' || !is_dir($path)) continue;

        if ($url === '' && AUTO_URL) {
            if ($webp === '') $webp = ($alias !== '') ? $alias : basename($path);
            $webp = safe_web_path($webp);
            if ($webp !== '') {
                $url = $baseUrl . '/' . $webp;
            }
        }

        $roots[] = build_local_root($alias, $path, $url);
    }
}

// fallback (최소 1개는 무조건)
if (count($roots) === 0) {
    // 우선순위: /mnt/storage → /opt/realcom-nas/DATA → /opt/realcom-nas → /
    $fallbackPath = '/mnt/storage';
    if (!is_dir($fallbackPath)) $fallbackPath = '/opt/realcom-nas/DATA';
    if (!is_dir($fallbackPath)) $fallbackPath = '/opt/realcom-nas';
    if (!is_dir($fallbackPath)) $fallbackPath = '/';

    $roots[] = build_local_root('DATA', $fallbackPath, '');
}

// -------------------------------------------------
// elFinder 실행
// -------------------------------------------------
$opts = array('roots' => $roots);
$connector = new elFinderConnector(new elFinder($opts));
$connector->run();

