# 경로/파일: realcom-nas-ui/modules/run_timeshift_policy.py
# 목적: cron/systemd에서 정책 1개 실행용 엔트리 (Timeshift)
# 사용:
#   /usr/bin/python3 /opt/realcom-nas/realcom-nas-ui/modules/run_timeshift_policy.py --id=xxxxxxxx
#   /usr/bin/python3 /opt/realcom-nas/realcom-nas-ui/modules/run_timeshift_policy.py --status
#   /usr/bin/python3 /opt/realcom-nas/realcom-nas-ui/modules/run_timeshift_policy.py --list

import os
import sys
import argparse
import datetime

# ✅ 크론에서 모듈 import 깨지는 거 방지: 프로젝트 루트(/opt/realcom-nas/realcom-nas-ui)를 sys.path에 추가
THIS_FILE = os.path.abspath(__file__)
PROJECT_ROOT = os.path.dirname(os.path.dirname(THIS_FILE))  # .../realcom-nas-ui
if PROJECT_ROOT not in sys.path:
    sys.path.insert(0, PROJECT_ROOT)

from modules import backup  # noqa: E402


def _now() -> str:
    return datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")


def main() -> int:
    ap = argparse.ArgumentParser()
    ap.add_argument("--id", help="policy id (8 chars)")
    ap.add_argument("--status", action="store_true", help="print timeshift status")
    ap.add_argument("--list", action="store_true", help="print timeshift snapshot list (raw)")
    args = ap.parse_args()

    try:
        if args.status:
            print(backup.timeshift_status())
            return 0

        if args.list:
            print(backup.list_snapshots_raw())
            return 0

        if not args.id:
            print("ERROR: --id is required (or use --status/--list)", file=sys.stderr)
            return 2

        pid = args.id.strip()
        # ✅ 실행
        msg = backup.run_policy(pid)
        # ✅ stdout은 cron 메일/저널에 남을 수 있으니 같이 출력
        print(f"[{_now()}] pid={pid} result={msg}")

        # ✅ 실패 문자열이면 exit 1로 끝내기
        if "실패" in msg or "오류" in msg:
            return 1
        return 0

    except Exception as e:
        print(f"[{_now()}] EXCEPTION: {e}", file=sys.stderr)
        return 1


if __name__ == "__main__":
    raise SystemExit(main())
