#!/usr/bin/env bash
set -e

APP_USER="realcom"
APP_HOME="/home/${APP_USER}"
APP_DIR="${APP_HOME}/realcom-nas-ui"
SERVICE_NAME="realcom-app"

echo "====================================="
echo "     RealNAS 올인원 설치 시작"
echo "====================================="

# -----------------------------------------
# 0. root 권한 체크
# -----------------------------------------
if [ "$EUID" -ne 0 ]; then
    echo "❌ root 권한으로 실행해야 합니다. (예: sudo ./install_realnas.sh)"
    exit 1
fi

# -----------------------------------------
# 1. ansible 설치
# -----------------------------------------
echo "[1/7] ansible 설치 중..."
apt update -y
apt install -y ansible python3 python3-venv python3-pip nginx samba

# -----------------------------------------
# 2. realcom 계정 생성
# -----------------------------------------
echo "[2/7] '${APP_USER}' 계정 확인 중..."
if id -u "${APP_USER}" >/dev/null 2>&1; then
    echo " - ${APP_USER} 이미 존재함"
else
    echo " - ${APP_USER} 계정 생성"
    useradd -m -s /bin/bash "${APP_USER}"
fi

mkdir -p "${APP_HOME}"
chown -R ${APP_USER}:${APP_USER} "${APP_HOME}"

# -----------------------------------------
# 3. NAS UI 소스 복사
# -----------------------------------------
echo "[3/7] NAS UI 소스 ${APP_DIR} 로 복사..."

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

if [ -d "${APP_DIR}" ]; then
    echo " - 기존 NAS UI 백업 생성"
    mv "${APP_DIR}" "${APP_DIR}.bak_$(date +%Y%m%d_%H%M%S)"
fi

mkdir -p "${APP_DIR}"

if command -v rsync >/dev/null 2>&1; then
    rsync -a "${SCRIPT_DIR}/" "${APP_DIR}/"
else
    cp -a "${SCRIPT_DIR}/." "${APP_DIR}/"
fi

chown -R ${APP_USER}:${APP_USER} "${APP_DIR}"

# -----------------------------------------
# 4. Ansible 플레이북 실행
# -----------------------------------------
echo "[4/7] Ansible 플레이북 실행..."

cd "${APP_DIR}"

if [ ! -f "inventory.ini" ]; then
    echo "❌ inventory.ini 파일이 없습니다!"
    exit 1
fi

if [ ! -f "realnas_setup.yml" ]; then
    echo "❌ realnas_setup.yml 파일이 없습니다!"
    exit 1
fi

ansible-playbook -i inventory.ini realnas_setup.yml

# -----------------------------------------
# 5. systemd 서비스 로드/재시작
# -----------------------------------------
echo "[5/7] Flask systemd 서비스 적용 중..."

systemctl daemon-reload
systemctl enable ${SERVICE_NAME}.service || true
systemctl restart ${SERVICE_NAME}.service || true

# -----------------------------------------
# 6. Nginx 프록시 설정 확인
# -----------------------------------------
echo "[6/7] Nginx reverse proxy 설정..."

if [ -f /etc/nginx/sites-available/realnas.conf ]; then
    ln -sf /etc/nginx/sites-available/realnas.conf /etc/nginx/sites-enabled/realnas.conf
    rm -f /etc/nginx/sites-enabled/default
    nginx -t && systemctl restart nginx
else
    echo "⚠ realnas.conf 파일이 ansible-playbook 에 의해 생성되지 않았습니다."
fi

# -----------------------------------------
# 7. 설치 완료 출력
# -----------------------------------------
echo "====================================="
echo "        RealNAS 설치 완료!"
echo "====================================="
echo "웹 UI 주소:   http://<서버IP>"
echo "Flask 서비스: realcom-app.service"
echo "Nginx 상태:    systemctl status nginx"
echo "서비스 로그:   journalctl -u realcom-app -f"
echo "====================================="
