import subprocess
import pkg_resources



# --- APT로 설치할 패키지들 ---
APT_PKGS = [
    "samba",              # 윈도우 파일 공유 서버 (SMB)
    "samba-common-bin",   # Samba 관리 명령어/툴
    "wireguard",          # WireGuard VPN 서버
    "rsync",              # 백업/동기화 툴
    "smartmontools",      # 디스크 상태(SMART)
    "ntfs-3g",            # NTFS 파일시스템 마운트
    "dosfstools",         # FAT/ExFAT 마운트/포맷
    "xfsprogs",           # XFS 파일시스템 툴
    "curl",               # 외부 네트워크/업데이트
    "avahi-daemon",       # 네트워크 검색 (Bonjour, mDNS)
    "cifs-utils",         # SMB 클라이언트 마운트
    "lsof",               # 열린 파일/프로세스 정보
    "lsblk",              # 블록 디바이스 정보
    "udisks2",            # 디스크/파티션 관리
    "parted",             # 파티션 툴
    "sudo"                # 루트 권한 명령 (스크립트 자동화)
]

# --- pip로 설치할 패키지들 ---
PIP_PKGS = [
    "flask",
    "markupsafe",
    "werkzeug",
    "qrcode",
    "pillow"
]

ALL_PKGS = APT_PKGS + PIP_PKGS


def check_pip_package(pkg):
    """pip 패키지 설치여부 & 버전 확인 (정확히)"""
    pkgs = {p.key: p.version for p in pkg_resources.working_set}
    return pkgs.get(pkg.lower()) or pkgs.get(pkg) or None

def check_package(pkg):
    if pkg == "lsblk":
        # lsblk는 util-linux에 포함되어 있으므로 직접 체크
        import shutil
        return shutil.which("lsblk") is not None
    elif pkg in PIP_PKGS:
        return check_pip_package(pkg) is not None
    else:
        result = subprocess.run(
            ["dpkg", "-s", pkg],
            capture_output=True, text=True)
        return "Status: install ok installed" in result.stdout


def check_all():
    """모든 패키지 설치여부 dict 반환"""
    return {pkg: check_package(pkg) for pkg in ALL_PKGS}

def install_package(pkg):
    """패키지 설치"""
    if pkg in PIP_PKGS:
        result = subprocess.run(
            ["sudo", "pip3", "install", pkg, "--break-system-packages"],
            capture_output=True, text=True)
    else:
        result = subprocess.run(
            ["sudo", "apt-get", "install", "-y", pkg],
            capture_output=True, text=True)
    ok = (result.returncode == 0)
    output = result.stdout + "\n" + result.stderr if result.stderr else result.stdout
    return ok, output

def install_all():
    """전부 설치"""
    output_all = []
    for pkg in ALL_PKGS:
        ok, out = install_package(pkg)
        output_all.append(f"[{pkg}] {'성공' if ok else '실패'}\n{out}\n")
    return "\n".join(output_all)

# 참고: 설치마법사에서
# status = check_all()
# success, output = install_package('smartmontools')  # 예시
