import os
import subprocess

# --- 공유폴더 정보 읽기 위한 share.py import ---
# 반드시 같은 디렉토리에 share.py가 있어야 함!
from share import list_share

# --- Nginx WebDAV conf 경로 설정 (수정 가능) ---
NGINX_CONF_PATH = "/etc/nginx/sites-available/webdav-shares.conf"
NGINX_LINK_PATH = "/etc/nginx/sites-enabled/webdav-shares.conf"
NGINX_PORT = 8000  # 사용할 포트 (필요에 따라 수정)

def make_webdav_location_block(share):
    """
    공유폴더 하나당 Nginx location 블럭 생성
    :param share: dict, list_share() 결과의 요소
    :return: str, location block
    """
    share_name = share['name']
    share_path = share['path']
    if not share_name or not share_path:
        return ""
    return f"""
    location /{share_name}/ {{
        alias {share_path}/;
        dav_methods     PUT DELETE MKCOL COPY MOVE;
        dav_ext_methods PROPFIND OPTIONS;
        dav_access      user:rw group:rw all:r;

       
        

        client_max_body_size 2G;
        create_full_put_path on;
        autoindex on;
    }}
    """

def generate_nginx_conf(shares):
    """
    공유폴더 리스트로 Nginx WebDAV conf 전체 생성
    """
    locations = [make_webdav_location_block(s) for s in shares if s.get('path')]
    return f"""
server {{
    listen {NGINX_PORT};
    server_name _;

    # WebDAV 공유폴더 목록 자동생성
    {"".join(locations)}
}}
"""

def save_and_reload_nginx(conf_text):
    """
    Nginx conf 저장하고 reload까지 자동수행
    """
    # conf 저장
    with open(NGINX_CONF_PATH, "w") as f:
        f.write(conf_text)
    # 심볼릭링크 생성(없으면)
    if not os.path.exists(NGINX_LINK_PATH):
        os.symlink(NGINX_CONF_PATH, NGINX_LINK_PATH)
    # Nginx 설정 문법 확인
    subprocess.run(["sudo", "nginx", "-t"], check=True)
    # Nginx 리로드
    subprocess.run(["sudo", "systemctl", "reload", "nginx"], check=True)
    print("[OK] Nginx WebDAV conf 갱신 & reload 완료!")

def main():
    shares = list_share()  # share.py에서 공유폴더 리스트 받아옴
    print(f"[INFO] {len(shares)}개 공유폴더 불러옴")
    conf_text = generate_nginx_conf(shares)
    print("[INFO] Nginx conf 미리보기 ↓↓↓\n")
    print(conf_text)
    save_and_reload_nginx(conf_text)

if __name__ == "__main__":
    main()
