# modules/nginx_user.py

import os
import subprocess

HTPASSWD_FILE = "/etc/nginx/.htpasswd"  # Nginx가 실제로 참조하는 htpasswd 파일 경로

def list_nginx_users():
    """
    htpasswd 파일에 등록된 사용자 목록 반환
    """
    users = []
    if not os.path.exists(HTPASSWD_FILE):
        return users
    with open(HTPASSWD_FILE, "r") as f:
        for line in f:
            if ':' in line and not line.startswith('#'):
                user = line.split(':')[0].strip()
                users.append(user)
    return users

def add_nginx_user(username, password):
    """
    새로운 사용자 추가 (중복 아이디는 불가)
    """
    if not username or not password:
        return False, "사용자명과 비밀번호는 필수입니다."
    if username in list_nginx_users():
        return False, "이미 존재하는 사용자명입니다."
    cmd = ['htpasswd', '-b', HTPASSWD_FILE, username, password]
    try:
        subprocess.run(cmd, check=True)
        return True, f"사용자 '{username}'가 추가되었습니다."
    except subprocess.CalledProcessError as e:
        return False, f"사용자 추가 실패: {e}"

def remove_nginx_user(username):
    """
    사용자 삭제
    """
    if username not in list_nginx_users():
        return False, "존재하지 않는 사용자입니다."
    try:
        lines = []
        with open(HTPASSWD_FILE, 'r') as f:
            lines = f.readlines()
        with open(HTPASSWD_FILE, 'w') as f:
            for line in lines:
                if not line.startswith(username + ':'):
                    f.write(line)
        return True, f"사용자 '{username}'가 삭제되었습니다."
    except Exception as e:
        return False, f"사용자 삭제 실패: {e}"

def change_nginx_user_password(username, password):
    """
    사용자 비밀번호 변경
    """
    if username not in list_nginx_users():
        return False, "존재하지 않는 사용자입니다."
    if not password:
        return False, "비밀번호는 필수입니다."
    cmd = ['htpasswd', '-b', HTPASSWD_FILE, username, password]
    try:
        subprocess.run(cmd, check=True)
        return True, f"사용자 '{username}' 비밀번호가 변경되었습니다."
    except subprocess.CalledProcessError as e:
        return False, f"비밀번호 변경 실패: {e}"
